// CURVEFIT.CPP
// Using Optivec C-library (www.optivec.com)

#include <champ.h>
#include <vdstd.h>  // Optivec
#include <vdmath.h> // ditto
#include <mdstd.h>  // ditto

NEWMATHERR  // Because Optivec is used with a 16-bit program

// Function declarations
void drawCurve ( dVector y, dVector x, int color, bool dots, bool clear );
void modelFunc ( dVector y, dVector x, unsigned size );

 // Globals
const unsigned sizex   = 100;
const unsigned polydeg = 3;
double fitCoeff[polydeg+1];   // For polynomial fit
double fitParam[2];           // For nonlinear fit
int parStatus[7];

void gmain ()
{
   ginit( "Curvefit" );
   gwindow( -0.1, 1.1, -1.1, 1.1 );
   cinit( "Curvefit", 22, 65, CPPosition( 400, 100 ) );
   cout << "Suppose your (unknown) theoretical data points" << endl
        << "come from a simple harmonic oscillation." << endl;

   dVector xExp  = VD_vector( sizex );
   dVector yExp  = VD_vector( sizex );
   dVector yFit  = VD_vector( sizex );

   VD_ramp( xExp, sizex, 0, 1.0/(sizex-1) );    // x-axis from 0 to 1
   VDx_sin( yExp, xExp, sizex, 2*M_PI, 0, 1 );  // Fake the theoretical model: sine
   drawCurve( yExp, xExp, BLACK, false, true );

   cout << "We make a measurement. This is like adding some noise." << endl;
   VD_noise( yFit, sizex, 1, 0.1 );
   VD_addV( yExp, yExp, yFit, sizex );
   gclear( WHITE );
   drawCurve( yExp, xExp, BLACK, true, true );

   cout << "We try to fit a polynomial of degree " << polydeg << endl;
   VD_polyfit( fitCoeff, polydeg, xExp, yExp, sizex );
   VD_poly( yFit, xExp, sizex, fitCoeff, polydeg ); // calculate fit curve
   cout << "Coefficients of fitting polynom: " << endl
        << "a0: " << fitCoeff[0] << endl
        << "a1: " << fitCoeff[1] << endl
        << "a2: " << fitCoeff[2] << endl
        << "a3: " << fitCoeff[3] << endl
        << "a4: " << fitCoeff[4] << endl
        << "a5: " << fitCoeff[5] << endl;
   drawCurve( yFit, xExp, RED, false, false );

   cout << "We make a nonlinear fit to the model function y = A*sin(2*pi*f*t)," << endl
        << "where amplitude A and frequency f are the fitting parameters." << endl;

   int nbParams = 2;
   // We must have a good initial guess for the parameters:
   fitParam[0] = 0.5; fitParam[1] = 0.7;
   parStatus[0] = parStatus[1] = 1;   // Enable fitting of parameters 0 and 1
   // The nonlinear fit will use the provided modelFunc
   // and set the global parameters fitParam
   VD_nonlinfit( fitParam, parStatus, nbParams,
                         xExp, yExp, sizex,
                         modelFunc,
                         NULL );
   cout << "Parameters from fitting: " << endl
        << "A: " << fitParam[0] << endl
        << "f: " << fitParam[1] << endl;
   modelFunc( yFit, xExp, sizex );    // Copy fitted values into yFit
   drawCurve( yFit, xExp, BLUE, false, false );

   V_freeAll();
   cend();
   gend();
}

void modelFunc ( dVector y, dVector x, unsigned size )
{
   for ( unsigned i = 0; i < size; i++ )
      y[i] = fitParam[0] * sin( 2 * M_PI * fitParam[1] * x[i] );
}

void drawCurve ( dVector y, dVector x, int color, bool dots, bool clear )
{
   if ( clear )
   {
      gclear();
      gpenColor( BLACK );
      gline( 0, 0, 1, 0 );
      gline( 0, -1.1, 0, 1.1 );
   }
   gpenColor( color );
   for ( int i = 0; i < sizex; i++ )
   {
      gdotSize( 0.005, 0.05 );
      if ( dots )
         gdot( x[i], y[i], color );
      else
         if ( i == 0 )
            gpos( x[0], y[0] );
         else
            gdraw( x[i], y[i] );
   }
    cout << "Press any key..." << endl;
    getch();
}


