// SOUND5.CPP
// Schnelle Datenacquisition mit einer Sound-Karte
// Kontinuierliche Aufnahme

#define SNDVERSION "V 1.12"

#include <champ.h>
#include <cppath.h>
#include <string.h>
#include <stdlib.h>
#include <process.h>
#include "sound4.rh"
#include <mmsystem.h>   // Must be included after champ.h
#include "cpfft.h"
#include "cpaudio.h"
#include "cpcropix.h"


// #define DEBUG


enum { stopped, recording, aborting } state = stopped;

// Global function declarations
void do_start ();
void do_stop ();
void do_quit ();
void initGlobals ();

// ---------------------------- Globals ------------------------------

long samplingFreq;   // in Hz
long samplingTime;   // in msec

// Number of samples we start getting the data from actual position
// Note: on certain notebooks we must go back quite a bit due to timing restrictions
long readOffset;

unsigned nbTimeSamples;  // Number of samples in time display, maximum 2000
unsigned char timeAry[2000];

// FFT
int filterType;
int fftOrder;   // power of 2, maximum 2048
float fftAry[2049];

void gmain ()
{
#ifdef DEBUG
	cinit( "Debug", 30, 30, CPPosition( 600, 100 ) );
#endif

   initGlobals();


	long pos;
	int i;

   CPMenu menu( "SMENU" );
   menu.registerMenuItem( IDM_START, do_start );
   menu.registerMenuItem( IDM_STOP, do_stop );
   menu.registerMenuItem( IDM_QUIT, do_quit );

   char buf[80];
   strcpy( buf, "Sound5 - " );
   strcat( buf, SNDVERSION );
   CPFrame f( buf, menu, CPSize( 600, 600 ) );

   CroPix timeCro (
      "Zeitfunktion",        // Window title
      10, 10,                // Window upper left coordinates
      nbTimeSamples, 255,    // Window width, height
      "25 msec",             // Unit x-axis
      "(Volt)",              // Unit y-Achse
      2,                     // Number of x gridline sections
      2,                     // Number of y gridline sections
      10,                    // Number of x tick sections
      2,                     // Number of y tick sections
      BLACK,                 // Window color
      YELLOW,                // Grid color
      WHITE );               // Trace color

   CroPix fftCro (
      "Spektrum",            // Window title
		10, 320,               // Window upper left coordinates
      fftOrder/2, 100,       // Window width, height
      "10 kHz",              // Unit x-axis
      "(Volt)",              // Unit y-Achse
      10,                    // Number of x gridline sections
      5,                     // Number of y gridline sections
      20,                    // Number of x tick sections
      5,                     // Number of y tick sections
      LIGHTGRAY,             // Window color
      BLACK,                 // Grid color
      BLACK,                 // Trace color
      CroPix::sticks );      // Trace type

   CPFft fft( fftAry, fftOrder, samplingFreq, filterType );
   CPAudio audio( samplingFreq, samplingTime, timeCro );
   timeCro.attach( audio );

   bool first = true;
   do
   {
      switch ( state )
      {
			case stopped:
            audio.stopRecording();
            first = true;
				CP::yield();
            break;

         case recording:
            CP::yield();

            if ( first )
            {
               audio.startRecording( true );   // Continuous recording
               first = false;
            }

				pos = audio.getPosition();

//    			cout << pos << " ";
				if ( pos < readOffset  )
					break;

				// Get data
				for ( i = 0; i < fftOrder; i++ )
				{
					fftAry[i+1] = (float)audio.aryData[pos  - readOffset + i];
				   if ( i < nbTimeSamples )
   					timeAry[i] = (unsigned char)audio.aryData[pos - readOffset + i];
				}

            fft.transform();
            fftCro.showTraceFloat( fftAry, fftOrder/2 + 1 );
            timeCro.showTrace( timeAry, nbTimeSamples );
            break;
      }
   } while ( state != aborting && timeCro.hwnd() != 0 );
   // When closing the window, hwnd becomes 0
   // Most cleanup is done by destructors

	cend();
}


void do_start ()
{
   state = recording;
}

void do_stop ()
{
   state = stopped;
}

void do_quit ()
{
   state = aborting;
}

void initGlobals ()
{
   int len;
   char buf[256];
   CPPath iniFile;
   iniFile.getCurrentDriveDir();
   iniFile.setFileExt( "SOUND5.INI" );
   if ( !iniFile.isFile() )
   {
      CP::msgBox( "Fatal error" ) << "Ini file " << iniFile.get() << " does not exist.";
      exit( 1 );
   }

   len = GetPrivateProfileString( "FFT", "FFTOrder", "", buf, sizeof( buf ), iniFile.get() );
   if ( len == 0 )
      strcpy( buf, "1024" );
   fftOrder = atoi( buf );

   len = GetPrivateProfileString( "FFT", "FilterType", "", buf, sizeof( buf ), iniFile.get() );
   if ( len == 0 )
      strcpy( buf, "1" );
   filterType = atoi( buf );

   len = GetPrivateProfileString( "Timing", "SamplingFreq", "", buf, sizeof( buf ), iniFile.get() );
   if ( len == 0 )
      strcpy( buf, "22050" );
   samplingFreq = atol( buf );

   len = GetPrivateProfileString( "Timing", "SamplingTime", "", buf, sizeof( buf ), iniFile.get() );
   if ( len == 0 )
      strcpy( buf, "50" );
   samplingTime = atol( buf );

   len = GetPrivateProfileString( "Timing", "ReadOffset", "", buf, sizeof( buf ), iniFile.get() );
   if ( len == 0 )
      strcpy( buf, "2000" );
   readOffset = atol( buf );

   len = GetPrivateProfileString( "Display", "NbtimeSamples", "", buf, sizeof( buf ), iniFile.get() );
   if ( len == 0 )
      strcpy( buf, "500" );
   nbTimeSamples = atoi( buf );

#ifdef DEBUG
   cout << "fftOrder: " << fftOrder << endl;
   cout << "filterType: " << filterType << endl;
   cout << "samplingTime: " << samplingTime << endl;
   cout << "samplingFreq: " << samplingFreq << endl;
   cout << "readOffset: " << readOffset << endl;
   cout << "nbTimeSamples: " << nbTimeSamples << endl;
#endif
}


