// SOUND4.CPP
// Schnelle Datenacquisition mit einer Sound-Karte

// Daten werden in einen CPHugeArrayChar eingelesen, obschon ein
// kleinerer Array fr diese Anwendung auch gengen wrde. Der Overhead
// ist im Vergleich zur FFT und Graphik klein.
// Vorteile: - Memoryallocation Freigabe werden automatisch durchgefhrt
//           - Auch grosse Samples mit demselben Algorithmus einlesbar

// Fr die FFT werden die Daten nachher in einen Float-Array bertragen, da dieser
// in der verwendeten Bibliothek der FFT-Routinen bentigt wird.

#define SNDVERSION "V 1.12"

#include <champ.h>
#include <cppath.h>
#include <stdlib.h>
#include <process.h>
#include <string.h>
#include "sound4.rh"
#include <mmsystem.h>   // Must be included after champ.h
#include "cpfft.h"
#include "cpaudio.h"
#include "cpcropix.h"

// #define DEBUG

enum { stopped, recording, aborting } state = stopped;

// Global function declarations
void do_start ();
void do_stop ();
void do_quit ();
void initGlobals ();

// ---------------------------- Globals ------------------------------

long samplingFreq;  // in Hz
long samplingTime;  // in msec
unsigned nbTimeSamples;  // Number of samples in time display, maximum 2000
unsigned char timeAry[2000];

// Ignore first samples gives better results on certain notebooks
unsigned startOffset;

// FFT
int filterType; // Hamming
int fftOrder;   // power of 2, maximum 2048
float fftAry[2049];

void gmain ()
{
#ifdef DEBUG
   cinit( "Debug", 30, 30, CPPosition( 600, 100 ) );
#endif
   initGlobals();  // Get values from INI file


   int long nbSamples = samplingFreq * samplingTime / 1000;
   if ( nbSamples < fftOrder + startOffset )
   {
      CP::msgBox( "Fatal error" ) << "Sampling time too small";
      exit( 0 );
   }

   int i;
   CPMenu menu( "SMENU" );
   menu.registerMenuItem( IDM_START, do_start );
   menu.registerMenuItem( IDM_STOP, do_stop );
   menu.registerMenuItem( IDM_QUIT, do_quit );

   char buf[80];
   strcpy( buf, "Sound4 - " );
   strcat( buf, SNDVERSION );
   CPFrame f( buf, menu, CPSize( 600, 600 ) );

   CroPix timeCro (
      "Zeitfunktion",        // Window title
      10, 10,                // Window upper left coordinates
      nbTimeSamples, 255,    // Window width, height
      "25 msec",             // Unit x-axis
      "(Volt)",              // Unit y-Achse
      2,                     // Number of x gridline sections
      2,                     // Number of y gridline sections
      10,                    // Number of x tick sections
      2,                     // Number of y tick sections
      BLACK,                 // Window color
      YELLOW,                // Grid color
      WHITE );               // Trace color

   CroPix fftCro (
      "Spektrum",            // Window title
      10, 320,               // Window upper left coordinates
      fftOrder/2, 100,       // Window width, height
      "10 kHz",              // Unit x-axis
      "(Volt)",              // Unit y-Achse
      10,                    // Number of x gridline sections
      5,                     // Number of y gridline sections
      20,                    // Number of x tick sections
      5,                     // Number of y tick sections
      LIGHTGRAY,             // Window color
      BLACK,                 // Grid color
      BLACK,                 // Trace color
      CroPix::sticks );      // Trace type

   CPFft fft( fftAry, fftOrder, samplingFreq, filterType );
   CPAudio audio( samplingFreq, samplingTime, timeCro );
   timeCro.attach( audio );

   bool first = true;
   do
   {
      switch ( state )
      {
         case stopped:
            audio.stopRecording();
            first = true;
            CP::yield();
            break;

         case recording:
            CP::yield();
            if ( first )
            {
               audio.startRecording();
               first = false;
               break;
            }

            while ( audio.isRecording() )
            {
               CP::yield();
               if ( timeCro.hwnd() == 0 )  // Window closed
                  break;
            }

            // Get data, ignore first startOffset samples
            for ( i = 0; i < fftOrder; i++ )
            {
               fftAry[i+1] = (float)audio.aryData[i+startOffset];
               if  ( i < nbTimeSamples )
                  timeAry[i] = (unsigned char)audio.aryData[i+startOffset];
            }

            // and restart recording immediately
            audio.startRecording();

            fft.transform();
            fftCro.showTraceFloat( fftAry, fftOrder/2 + 1 );
            timeCro.showTrace( timeAry, nbTimeSamples );
            break;
      }
   } while ( state != aborting && timeCro.hwnd() != 0 );
   // When closing the window, hwnd becomes 0
   // Most cleanup is done by destructors
   cend();
}


void do_start ()
{
   state = recording;
}

void do_stop ()
{
   state = stopped;
}

void do_quit ()
{
   state = aborting;
}

void initGlobals ()
{
   int len;
   char buf[256];
   CPPath iniFile;
   iniFile.getCurrentDriveDir();
   iniFile.setFileExt( "SOUND4.INI" );
   if ( !iniFile.isFile() )
   {
      CP::msgBox( "Fatal error" ) << "Ini file " << iniFile.get() << " does not exist.";
      exit( 1 );
   }

   len = GetPrivateProfileString( "FFT", "FFTOrder", "", buf, sizeof( buf ), iniFile.get() );
   if ( len == 0 )
      strcpy( buf, "1024" );
   fftOrder = atoi( buf );

   len = GetPrivateProfileString( "FFT", "FilterType", "", buf, sizeof( buf ), iniFile.get() );
   if ( len == 0 )
      strcpy( buf, "1" );
   filterType = atoi( buf );

   len = GetPrivateProfileString( "Timing", "SamplingFreq", "", buf, sizeof( buf ), iniFile.get() );
   if ( len == 0 )
      strcpy( buf, "22050" );
   samplingFreq = atol( buf );

   len = GetPrivateProfileString( "Timing", "SamplingTime", "", buf, sizeof( buf ), iniFile.get() );
   if ( len == 0 )
      strcpy( buf, "50" );
   samplingTime = atol( buf );

   len = GetPrivateProfileString( "Timing", "StartOffset", "", buf, sizeof( buf ), iniFile.get() );
   if ( len == 0 )
      strcpy( buf, "20" );
   startOffset = atoi( buf );

   len = GetPrivateProfileString( "Display", "NbtimeSamples", "", buf, sizeof( buf ), iniFile.get() );
   if ( len == 0 )
      strcpy( buf, "500" );
   nbTimeSamples = atoi( buf );

#ifdef DEBUG
   cout << "fftOrder: " << fftOrder << endl;
   cout << "filterType: " << filterType << endl;
   cout << "samplingTime: " << samplingTime << endl;
   cout << "samplingFreq: " << samplingFreq << endl;
   cout << "startOffset: " << startOffset << endl;
   cout << "nbTimeSamples: " << nbTimeSamples << endl;
#endif
}


