// CPFFT.CPP

#define SWAP(a,b) tempr=(a);(a)=(b);(b)=tempr
#define delta 10

#include <alloc.h>
#include <process.h>
#include <champ.h>
#include "cpfft.h"

CPFft::CPFft ( float * fftAry, int fftOrder, float samplingFreq, int filterType,
               bool logScale )
   : _fftOrder( fftOrder ),
     _samplingFreq( samplingFreq ),
     _filterType( filterType ),
     _logScale( logScale ),
     _fftAry ( fftAry )
{
   _normAry = new float[ fftOrder/2 + 1 ];
}

CPFft::~CPFft ()
{
  delete [] _normAry;
}

/* ---------------------------------------------------------------------- */
/*                      Fourier transform functions                       */
/* ---------------------------------------------------------------------- */
void CPFft::four1 ( float data[], int nn, int isign )
/* Complex fast fourier transform routine, see
   Press, Flannery, Teukolsky, Vetterling, Numerical Recipies in C,
   Cambridge, page 398 ff.

   Input data in array data[1..2*nn], where nn is the order
   of the FFT (must be an order of 2, typically 256, 512, 1024, ...)
   in pairs of complex samples at nn consecutive times 0, dt, 2*dt, ...

   time:     0        dt          ...     (nn-1)*dt
   real:     data[1]  data[3]     ...     data[2*nn-1]
   imag:     data[2]  data[4]     ...     data[2*nn]

   isign = 1 for FFT, isign = -1 for inverse FFT

   Output data in same array, with frequency step df = (1 / nn) * (1 / dt)

   freq:     0        df        ... (1/2) * 1/dt  ...   -df
   real:     data[1]  data[3]       data[nn+1]          data[2*nn-1]
   imag:     data[2]  data[4]       data[nn+2]          data[2*nn]

   e.g. for dt = 25 us, fsampling = 40 kHz, nn = 512:

   Input:
   -----
   sample #: 1        2                   512
   time:     0        25 us       ...     12.775 ms
   real:     data[1]  data[3]     ...     data[2*nn-1]
   imag:     data[2]  data[4]     ...     data[2*nn]

   Output:
   ------
   freq:     0        78Hz    19.922kHz  20 kHz    -19.922kHz ...  -78 Hz
   real:     data[1]  data[3]            data[nn+1]           ... data[2*nn-1]
   imag:     data[2]  data[4]            data[nn+2]           ... data[2*nn]
*/

{
int n, mmax, m, j, istep, i;
double wtemp, wr, wpr, wpi, wi, theta;
float tempr, tempi;

        n = nn << 1;
        j = 1;
        for ( i=1; i<n; i+=2 ) {
                if ( j > i ) {
                        SWAP(data[j], data[i]);
                        SWAP(data[j+1], data[i+1]);
                }
                m = n >> 1;

                while ( m >= 2 && j > m ) {
                        j -= m;
                        m >>= 1;
                }
                j += m;
        }
        mmax = 2;
        while ( n > mmax ) {
                istep = 2*mmax;
                theta = 2*M_PI/(isign*mmax);
                wtemp = sin(0.5*theta);
                wpr = -2.0*wtemp*wtemp;
                wpi = sin(theta);
                wr = 1.0;
                wi = 0.0;
                for ( m=1; m<mmax; m+=2 ) {
                        for ( i=m; i<=n; i+=istep ) {
                                j = i+mmax;
                                tempr = wr*data[j] - wi*data[j+1];
                                tempi = wr*data[j+1] + wi*data[j];
                                data[j] = data[i] - tempr;
                                data[j+1] = data[i+1] - tempi;
                                data[i] += tempr;
                                data[i+1] += tempi;
                        }
                        wr = (wtemp=wr)*wpr - wi*wpi + wr;
                        wi = wi*wpr + wtemp*wpi + wi;
                }
                mmax = istep;
        }
}

void CPFft::realft ( float data[], int n, int isign )
/* Real fast fourier transform routine, see
   Press, Flannery, Teukolsky, Vetterling, Numerical Recipies in C,
   Cambridge, page 398 ff.

   n = nn/2, where nn is the order of the FFT
   (nn must be an order of 2, typically 256, 512, 1024, ...)

   time:     0        dt          ...     (nn-1)*dt
   input:    data[1]  data[2]     ...     data[nn]

   isign = 1 for FFT, isign = -1 for inverse FFT

   Output data in same array, with frequency step df = (1 / n) * (1 / dt)

   freq:   0          df      ...   (1/2) * 1/dt - df   (1/2) * 1/dt
   real:   data[1]    data[3]       data[nn-1]           data[2]
   imag:   (always 0) data[4]       data[nn]             (always 0)

   e.g. for dt = 25 us, fsampling = 40 kHz, nn = 512

   Input:
   -----
   sample #: 1        2                   512
   time:     0        25 us       ...     12.775 ms
   data:     data[1]  data[3]     ...     data[512]

   Output:
   ------
   freq #:   0(Offset) 1(Basefreq)        255         256
   freq:     0Hz       78Hz       ...     19.922kHz   20kHz
   real:     data[1]   data[3]    ...     data[511]   data[2]
   imag:      --       data[4]    ...     data[512]     --
*/

{
int i, i1, i2, i3, i4, n2p3;
float c1=0.5, c2, h1r, h1i, h2r, h2i;
double wr, wi, wpr, wpi, wtemp, theta;

        theta=M_PI/(double) n;
        if ( isign == 1 ) {
                c2 = -0.5;
                four1(data, n, 1);
        } else {
                c2 = 0.5;
                theta = -theta;
        }
        wtemp = sin(0.5*theta);
        wpr = -2.0*wtemp*wtemp;
        wpi = sin(theta);
        wr = 1.0+wpr;
        wi = wpi;
        n2p3 = 2*n + 3;
        for ( i=2; i<=n/2; i++ ) {
                i4 = 1+(i3=n2p3-(i2=1+(i1=i+i-1)));
                h1r = c1*(data[i1] + data[i3]);
                h1i = c1*(data[i2] - data[i4]);
                h2r = -c2*(data[i2] + data[i4]);
                h2i = c2*(data[i1] - data[i3]);
                data[i1] = h1r+wr*h2r - wi*h2i;
                data[i2] = h1i + wr*h2i + wi*h2r;
                data[i3] = h1r - wr*h2r + wi*h2i;
                data[i4] = -h1i + wr*h2i + wi*h2r;
                wr = (wtemp=wr)*wpr - wi*wpi + wr;
                wi = wi*wpr + wtemp*wpi + wi;
        }
        if ( isign == 1 ) {
                data[1] = (h1r=data[1]) + data[2];
                data[2] = h1r-data[2];
        } else {
                data[1] = c1*((h1r=data[1]) + data[2]);
                data[2] = c1*(h1r-data[2]);
                four1(data, n, -1);
        }
}

long CPFft::fdiv ( double a, double b )
// Rounded division
{
double div;

   div = a / b;
   if ( div - floor(div) < ceil(div) - div ) return((long)floor(div));
   else return((long)ceil(div));
}

void CPFft::transform ()
{
   CP::yield();
   float signalmax = 0;
   float lowLevel = 10;
   float signal;
   int i;

   // Determine maximum of data input to check if we want to transform
   for ( i = 1; i <= _fftOrder; i++ )
      signal = _fftAry[i] > 127 ? _fftAry[i] - 127 : 127 - _fftAry[i];
      if (  signal > signalmax )
         signalmax = signal;

   if (  signalmax < lowLevel )
   {
      for ( i = 0; i <= _fftOrder/2; i++ )
         _fftAry[i] = 100;
//      cout << "input signal too small" << endl;
      return;
   }

   float filter_freq, spectral_resolution;
   int filterlinenb;
   float linemax = 0;

   spectral_resolution = _samplingFreq / _fftOrder;
   filter_freq = _samplingFreq / _fftOrder;
   filterlinenb = (int)fdiv(filter_freq, spectral_resolution);

    // Filter  data
   if ( _filterType == 1 ) // Hamming
      for ( i = 1; i <= _fftOrder; i++ )
         _fftAry[i] = _fftAry[i] * ( 0.54 - 0.46*cos(2*M_PI*i/_fftOrder) );

   if ( _filterType == 2 ) // Hanning
      for ( i = 1; i <= _fftOrder; i++ )
         _fftAry[i] = _fftAry[i] * ( 0.50 - 0.50*cos(2*M_PI*i/_fftOrder) );

   // Perform FFT
   CP::yield();
   realft( _fftAry, _fftOrder / 2, 1 );
   CP::yield();

   // Calculate absolute values
   for ( i = 1; i <= _fftOrder/2-1; i++ )
      _normAry[i] = sqrt( (_fftAry[2*i+1]*_fftAry[2*i+1]) +
                      (_fftAry[2*i+2]*_fftAry[2*i+2]) );

// Do not use coponent at f = 0 Hz  (DC offset)
//   norm[0] = fabs(fftdata[1]);
//   if ( norm[0] > linemax ) linemax = norm[0];

   _normAry[0] = 0;
   _normAry[_fftOrder/2] = fabs( _fftAry[2] );

   // Mask spectral components due to filter
   if ( _filterType ) {
      for ( i = 0; i <= filterlinenb+2; i++ )
         _normAry[i] = 0;
   }

   // Determine maximum spectral component
   for ( i = 0; i <= _fftOrder/2; i++ )
      if ( _normAry[i] > linemax )
         linemax = _normAry[i];

   // Normalize spectrum to 0..100 ( if not completely 0 )and invert for display
   if ( linemax > 0 )
   {
      for ( i = 0; i <= _fftOrder/2; i++ )
         _normAry[i] = 100 - 100*_normAry[i] / linemax;

      // Convert to logarithmic scale (dB)
      if ( _logScale )
         for ( i = 0; i <= _fftOrder/2; i++ )
            if ( _normAry[i] < 1E-3 )
               _normAry[i] = 0;
            else
               _normAry[i] = 60 + 20*log10( _normAry[i] );


      // Copy back to fftAry to have access from outside
      for ( i = 0; i <= _fftOrder/2; i++ )
         _fftAry[i] = _normAry[i];
   }
   else
   {
      for ( i = 0; i <= _fftOrder/2; i++ )
         _fftAry[i] = 100;
//      cout << "spectral components zero" << endl;
   }
}


